// Copyright 2017-2018 - Universite de Strasbourg/CNRS
// The CDS HEALPix library is developped by the Centre de Donnees
// astronomiques de Strasbourgs (CDS) from the following external papers:
//  - [Gorsky2005]     - "HEALPix: A Framework for High-Resolution Discretization and
//                       Fast Analysis of Data Distributed on the Sphere"
//                       http://adsabs.harvard.edu/abs/2005ApJ...622..759G
//  - [Calabretta2004] - "Mapping on the HEALPix grid"
//                       http://adsabs.harvard.edu/abs/2004astro.ph.12607C
//  - [Calabretta2007] - "Mapping on the HEALPix grid"
//                       http://adsabs.harvard.edu/abs/2007MNRAS.381..865C
//  - [Reinecke2015]   - "Efficient data structures for masks on 2D grids"
//                       http://adsabs.harvard.edu/abs/2015A&A...580A.132R
// It is distributed under the terms of the BSD License 2.0
//
// This file is part of the CDS HEALPix library.
//

package cds.healpix.fillingcurve;

/**
 * Implementation of the 3d z-order curve based on lookup tables.
 * 
 * @author F.-X. Pineau
 *
 */
public class ZOrderCurve3D implements FillingCurve3D {

  public static final ZOrderCurve3D INSTANCE = new ZOrderCurve3D();

  /** Lookup table storing the result of the {@link FillingCuve2D#i02hash(int)} method
   *  for all all possible unsigned bytes, so for all value in [0, 255].
   *  We tested the difference of performance between array of short and of long.
   *  In the first case, casts (into long) are required but the array is smaller so will occupy
   *  less space in the L1 cache. We were not able to detect any difference so we choose short.  */
  private static final int[] LUPT_TO_HASH = new int[] {
      0x000000, 0x000001, 0x000008, 0x000009, 0x000040, 0x000041, 0x000048, 0x000049, 0x000200,
      0x000201, 0x000208, 0x000209, 0x000240, 0x000241, 0x000248, 0x000249, 0x001000, 0x001001,
      0x001008, 0x001009, 0x001040, 0x001041, 0x001048, 0x001049, 0x001200, 0x001201, 0x001208,
      0x001209, 0x001240, 0x001241, 0x001248, 0x001249, 0x008000, 0x008001, 0x008008, 0x008009,
      0x008040, 0x008041, 0x008048, 0x008049, 0x008200, 0x008201, 0x008208, 0x008209, 0x008240,
      0x008241, 0x008248, 0x008249, 0x009000, 0x009001, 0x009008, 0x009009, 0x009040, 0x009041,
      0x009048, 0x009049, 0x009200, 0x009201, 0x009208, 0x009209, 0x009240, 0x009241, 0x009248,
      0x009249, 0x040000, 0x040001, 0x040008, 0x040009, 0x040040, 0x040041, 0x040048, 0x040049,
      0x040200, 0x040201, 0x040208, 0x040209, 0x040240, 0x040241, 0x040248, 0x040249, 0x041000,
      0x041001, 0x041008, 0x041009, 0x041040, 0x041041, 0x041048, 0x041049, 0x041200, 0x041201,
      0x041208, 0x041209, 0x041240, 0x041241, 0x041248, 0x041249, 0x048000, 0x048001, 0x048008,
      0x048009, 0x048040, 0x048041, 0x048048, 0x048049, 0x048200, 0x048201, 0x048208, 0x048209,
      0x048240, 0x048241, 0x048248, 0x048249, 0x049000, 0x049001, 0x049008, 0x049009, 0x049040,
      0x049041, 0x049048, 0x049049, 0x049200, 0x049201, 0x049208, 0x049209, 0x049240, 0x049241,
      0x049248, 0x049249, 0x200000, 0x200001, 0x200008, 0x200009, 0x200040, 0x200041, 0x200048,
      0x200049, 0x200200, 0x200201, 0x200208, 0x200209, 0x200240, 0x200241, 0x200248, 0x200249,
      0x201000, 0x201001, 0x201008, 0x201009, 0x201040, 0x201041, 0x201048, 0x201049, 0x201200,
      0x201201, 0x201208, 0x201209, 0x201240, 0x201241, 0x201248, 0x201249, 0x208000, 0x208001,
      0x208008, 0x208009, 0x208040, 0x208041, 0x208048, 0x208049, 0x208200, 0x208201, 0x208208,
      0x208209, 0x208240, 0x208241, 0x208248, 0x208249, 0x209000, 0x209001, 0x209008, 0x209009,
      0x209040, 0x209041, 0x209048, 0x209049, 0x209200, 0x209201, 0x209208, 0x209209, 0x209240,
      0x209241, 0x209248, 0x209249, 0x240000, 0x240001, 0x240008, 0x240009, 0x240040, 0x240041,
      0x240048, 0x240049, 0x240200, 0x240201, 0x240208, 0x240209, 0x240240, 0x240241, 0x240248,
      0x240249, 0x241000, 0x241001, 0x241008, 0x241009, 0x241040, 0x241041, 0x241048, 0x241049,
      0x241200, 0x241201, 0x241208, 0x241209, 0x241240, 0x241241, 0x241248, 0x241249, 0x248000,
      0x248001, 0x248008, 0x248009, 0x248040, 0x248041, 0x248048, 0x248049, 0x248200, 0x248201, 
      0x248208, 0x248209, 0x248240, 0x248241, 0x248248, 0x248249, 0x249000, 0x249001, 0x249008,
      0x249009, 0x249040, 0x249041, 0x249048, 0x249049, 0x249200, 0x249201, 0x249208, 0x249209,
      0x249240, 0x249241, 0x249248, 0x249249
  };

  private static final long[] LUPT_TO_IJK_INT = new long[] {
      0x00000000000L, 0x00000000001L, 0x00000100000L, 0x00000100001L, 0x10000000000L, 0x10000000001L,
      0x10000100000L, 0x10000100001L, 0x00000000002L, 0x00000000003L, 0x00000100002L, 0x00000100003L,
      0x10000000002L, 0x10000000003L, 0x10000100002L, 0x10000100003L, 0x00000200000L, 0x00000200001L,
      0x00000300000L, 0x00000300001L, 0x10000200000L, 0x10000200001L, 0x10000300000L, 0x10000300001L,
      0x00000200002L, 0x00000200003L, 0x00000300002L, 0x00000300003L, 0x10000200002L, 0x10000200003L,
      0x10000300002L, 0x10000300003L, 0x20000000000L, 0x20000000001L, 0x20000100000L, 0x20000100001L,
      0x30000000000L, 0x30000000001L, 0x30000100000L, 0x30000100001L, 0x20000000002L, 0x20000000003L,
      0x20000100002L, 0x20000100003L, 0x30000000002L, 0x30000000003L, 0x30000100002L, 0x30000100003L,
      0x20000200000L, 0x20000200001L, 0x20000300000L, 0x20000300001L, 0x30000200000L, 0x30000200001L,
      0x30000300000L, 0x30000300001L, 0x20000200002L, 0x20000200003L, 0x20000300002L, 0x20000300003L,
      0x30000200002L, 0x30000200003L, 0x30000300002L, 0x30000300003L, 0x00000000004L, 0x00000000005L,
      0x00000100004L, 0x00000100005L, 0x10000000004L, 0x10000000005L, 0x10000100004L, 0x10000100005L,
      0x00000000006L, 0x00000000007L, 0x00000100006L, 0x00000100007L, 0x10000000006L, 0x10000000007L,
      0x10000100006L, 0x10000100007L, 0x00000200004L, 0x00000200005L, 0x00000300004L, 0x00000300005L,
      0x10000200004L, 0x10000200005L, 0x10000300004L, 0x10000300005L, 0x00000200006L, 0x00000200007L,
      0x00000300006L, 0x00000300007L, 0x10000200006L, 0x10000200007L, 0x10000300006L, 0x10000300007L,
      0x20000000004L, 0x20000000005L, 0x20000100004L, 0x20000100005L, 0x30000000004L, 0x30000000005L,
      0x30000100004L, 0x30000100005L, 0x20000000006L, 0x20000000007L, 0x20000100006L, 0x20000100007L,
      0x30000000006L, 0x30000000007L, 0x30000100006L, 0x30000100007L, 0x20000200004L, 0x20000200005L,
      0x20000300004L, 0x20000300005L, 0x30000200004L, 0x30000200005L, 0x30000300004L, 0x30000300005L,
      0x20000200006L, 0x20000200007L, 0x20000300006L, 0x20000300007L, 0x30000200006L, 0x30000200007L,
      0x30000300006L, 0x30000300007L, 0x00000400000L, 0x00000400001L, 0x00000500000L, 0x00000500001L,
      0x10000400000L, 0x10000400001L, 0x10000500000L, 0x10000500001L, 0x00000400002L, 0x00000400003L,
      0x00000500002L, 0x00000500003L, 0x10000400002L, 0x10000400003L, 0x10000500002L, 0x10000500003L,
      0x00000600000L, 0x00000600001L, 0x00000700000L, 0x00000700001L, 0x10000600000L, 0x10000600001L,
      0x10000700000L, 0x10000700001L, 0x00000600002L, 0x00000600003L, 0x00000700002L, 0x00000700003L,
      0x10000600002L, 0x10000600003L, 0x10000700002L, 0x10000700003L, 0x20000400000L, 0x20000400001L,
      0x20000500000L, 0x20000500001L, 0x30000400000L, 0x30000400001L, 0x30000500000L, 0x30000500001L,
      0x20000400002L, 0x20000400003L, 0x20000500002L, 0x20000500003L, 0x30000400002L, 0x30000400003L,
      0x30000500002L, 0x30000500003L, 0x20000600000L, 0x20000600001L, 0x20000700000L, 0x20000700001L,
      0x30000600000L, 0x30000600001L, 0x30000700000L, 0x30000700001L, 0x20000600002L, 0x20000600003L,
      0x20000700002L, 0x20000700003L, 0x30000600002L, 0x30000600003L, 0x30000700002L, 0x30000700003L,
      0x00000400004L, 0x00000400005L, 0x00000500004L, 0x00000500005L, 0x10000400004L, 0x10000400005L,
      0x10000500004L, 0x10000500005L, 0x00000400006L, 0x00000400007L, 0x00000500006L, 0x00000500007L,
      0x10000400006L, 0x10000400007L, 0x10000500006L, 0x10000500007L, 0x00000600004L, 0x00000600005L,
      0x00000700004L, 0x00000700005L, 0x10000600004L, 0x10000600005L, 0x10000700004L, 0x10000700005L,
      0x00000600006L, 0x00000600007L, 0x00000700006L, 0x00000700007L, 0x10000600006L, 0x10000600007L,
      0x10000700006L, 0x10000700007L, 0x20000400004L, 0x20000400005L, 0x20000500004L, 0x20000500005L,
      0x30000400004L, 0x30000400005L, 0x30000500004L, 0x30000500005L, 0x20000400006L, 0x20000400007L,
      0x20000500006L, 0x20000500007L, 0x30000400006L, 0x30000400007L, 0x30000500006L, 0x30000500007L,
      0x20000600004L, 0x20000600005L, 0x20000700004L, 0x20000700005L, 0x30000600004L, 0x30000600005L,
      0x30000700004L, 0x30000700005L, 0x20000600006L, 0x20000600007L, 0x20000700006L, 0x20000700007L,
      0x30000600006L, 0x30000600007L, 0x30000700006L, 0x30000700007L, 0x40000000000L, 0x40000000001L,
      0x40000100000L, 0x40000100001L, 0x50000000000L, 0x50000000001L, 0x50000100000L, 0x50000100001L,
      0x40000000002L, 0x40000000003L, 0x40000100002L, 0x40000100003L, 0x50000000002L, 0x50000000003L,
      0x50000100002L, 0x50000100003L, 0x40000200000L, 0x40000200001L, 0x40000300000L, 0x40000300001L,
      0x50000200000L, 0x50000200001L, 0x50000300000L, 0x50000300001L, 0x40000200002L, 0x40000200003L,
      0x40000300002L, 0x40000300003L, 0x50000200002L, 0x50000200003L, 0x50000300002L, 0x50000300003L,
      0x60000000000L, 0x60000000001L, 0x60000100000L, 0x60000100001L, 0x70000000000L, 0x70000000001L,
      0x70000100000L, 0x70000100001L, 0x60000000002L, 0x60000000003L, 0x60000100002L, 0x60000100003L,
      0x70000000002L, 0x70000000003L, 0x70000100002L, 0x70000100003L, 0x60000200000L, 0x60000200001L,
      0x60000300000L, 0x60000300001L, 0x70000200000L, 0x70000200001L, 0x70000300000L, 0x70000300001L,
      0x60000200002L, 0x60000200003L, 0x60000300002L, 0x60000300003L, 0x70000200002L, 0x70000200003L,
      0x70000300002L, 0x70000300003L, 0x40000000004L, 0x40000000005L, 0x40000100004L, 0x40000100005L,
      0x50000000004L, 0x50000000005L, 0x50000100004L, 0x50000100005L, 0x40000000006L, 0x40000000007L,
      0x40000100006L, 0x40000100007L, 0x50000000006L, 0x50000000007L, 0x50000100006L, 0x50000100007L,
      0x40000200004L, 0x40000200005L, 0x40000300004L, 0x40000300005L, 0x50000200004L, 0x50000200005L,
      0x50000300004L, 0x50000300005L, 0x40000200006L, 0x40000200007L, 0x40000300006L, 0x40000300007L,
      0x50000200006L, 0x50000200007L, 0x50000300006L, 0x50000300007L, 0x60000000004L, 0x60000000005L,
      0x60000100004L, 0x60000100005L, 0x70000000004L, 0x70000000005L, 0x70000100004L, 0x70000100005L,
      0x60000000006L, 0x60000000007L, 0x60000100006L, 0x60000100007L, 0x70000000006L, 0x70000000007L,
      0x70000100006L, 0x70000100007L, 0x60000200004L, 0x60000200005L, 0x60000300004L, 0x60000300005L,
      0x70000200004L, 0x70000200005L, 0x70000300004L, 0x70000300005L, 0x60000200006L, 0x60000200007L,
      0x60000300006L, 0x60000300007L, 0x70000200006L, 0x70000200007L, 0x70000300006L, 0x70000300007L,
      0x40000400000L, 0x40000400001L, 0x40000500000L, 0x40000500001L, 0x50000400000L, 0x50000400001L,
      0x50000500000L, 0x50000500001L, 0x40000400002L, 0x40000400003L, 0x40000500002L, 0x40000500003L,
      0x50000400002L, 0x50000400003L, 0x50000500002L, 0x50000500003L, 0x40000600000L, 0x40000600001L,
      0x40000700000L, 0x40000700001L, 0x50000600000L, 0x50000600001L, 0x50000700000L, 0x50000700001L,
      0x40000600002L, 0x40000600003L, 0x40000700002L, 0x40000700003L, 0x50000600002L, 0x50000600003L,
      0x50000700002L, 0x50000700003L, 0x60000400000L, 0x60000400001L, 0x60000500000L, 0x60000500001L,
      0x70000400000L, 0x70000400001L, 0x70000500000L, 0x70000500001L, 0x60000400002L, 0x60000400003L,
      0x60000500002L, 0x60000500003L, 0x70000400002L, 0x70000400003L, 0x70000500002L, 0x70000500003L,
      0x60000600000L, 0x60000600001L, 0x60000700000L, 0x60000700001L, 0x70000600000L, 0x70000600001L,
      0x70000700000L, 0x70000700001L, 0x60000600002L, 0x60000600003L, 0x60000700002L, 0x60000700003L,
      0x70000600002L, 0x70000600003L, 0x70000700002L, 0x70000700003L, 0x40000400004L, 0x40000400005L,
      0x40000500004L, 0x40000500005L, 0x50000400004L, 0x50000400005L, 0x50000500004L, 0x50000500005L,
      0x40000400006L, 0x40000400007L, 0x40000500006L, 0x40000500007L, 0x50000400006L, 0x50000400007L,
      0x50000500006L, 0x50000500007L, 0x40000600004L, 0x40000600005L, 0x40000700004L, 0x40000700005L,
      0x50000600004L, 0x50000600005L, 0x50000700004L, 0x50000700005L, 0x40000600006L, 0x40000600007L,
      0x40000700006L, 0x40000700007L, 0x50000600006L, 0x50000600007L, 0x50000700006L, 0x50000700007L,
      0x60000400004L, 0x60000400005L, 0x60000500004L, 0x60000500005L, 0x70000400004L, 0x70000400005L,
      0x70000500004L, 0x70000500005L, 0x60000400006L, 0x60000400007L, 0x60000500006L, 0x60000500007L,
      0x70000400006L, 0x70000400007L, 0x70000500006L, 0x70000500007L, 0x60000600004L, 0x60000600005L,
      0x60000700004L, 0x60000700005L, 0x70000600004L, 0x70000600005L, 0x70000700004L, 0x70000700005L,
      0x60000600006L, 0x60000600007L, 0x60000700006L, 0x60000700007L, 0x70000600006L, 0x70000600007L,
      0x70000700006L, 0x70000700007L
  };


  private ZOrderCurve3D() { }

  @Override
  public long ijk2hash(int i, int j, int k) {
    return (i002hash(k) << 2) | (i002hash(j) << 1)  | i002hash(i); 
  }

  @Override
  public long i002hash(int i) { // max 19 bits (3 * 8 = 24) 
    assert (i == -1) || (i &  0xFF800000) == 0;
    return (((long) LUPT_TO_HASH[(i & 0x00070000) >>> 16]) << 48)
         | (((long) LUPT_TO_HASH[(i & 0x0000FF00) >>>  8]) << 24)
        |  ((long) LUPT_TO_HASH[ i & 0x000000FF]);
  }

  @Override
  public long hash2ijk(long hash) {
    return LUPT_TO_IJK_INT[(int) ((hash & 0x7FC0000000000000L) >>> 54)] << 18
         | LUPT_TO_IJK_INT[(int) ((hash & 0x003FE00000000000L) >>> 45)] << 15
         | LUPT_TO_IJK_INT[(int) ((hash & 0x00001FF000000000L) >>> 36)] << 12
         | LUPT_TO_IJK_INT[(int) ((hash & 0x0000000FF8000000L) >>> 27)] << 9
         | LUPT_TO_IJK_INT[(int) ((hash & 0x0000000007FC0000L) >>> 18)] << 6
         | LUPT_TO_IJK_INT[(int) ((hash & 0x000000000003FE00L) >>>  9)] << 3
         | LUPT_TO_IJK_INT[(int)  (hash & 0x00000000000001FFL)];
  }

  @Override
  public long hash2i00(long hash) {
    assert (0xFFFFFFFFFFFB6DB6L & hash) == 0;
    return hash2ijk(hash);
  }

  @Override
  public int ijk2i(long ijk) {
    return ((int) ijk) & 0x000FFFFF;
  }

  @Override
  public int ijk2j(long ijk) {
    return ((int) (ijk >> 20)) & 0x000FFFFF;
  }

  @Override
  public int ijk2k(long ijk) {
    return (int) (ijk >> 40);
  }

}
