import os
import subprocess
import sys
from io import BytesIO

from translate.storage import factory, mo
from translate.tools import pocompile

from . import test_base

MO_BIG_ENDIAN = b"\x95\x04\x12\xde\x00\x00\x00\x2a\x00\x00\x00\x02\x00\x00\x00\x1c\x00\x00\x00,\x00\x00\x00\x05\x00\x00\x00<\x00\x00\x00\x06\x00\x00\x00P\x00\x00\x00\x07\x00\x00\x00W\x00\x00\x00\x03\x00\x00\x00_\x00\x00\x00\x06\x00\x00\x00c\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02simple\x00unicode\x00Een\x00\xe2\x80\xa0wee\x00"
MO_LITTLE_ENDIAN = b"\xde\x12\x04\x95\x2a\x00\x00\x00\x02\x00\x00\x00\x1c\x00\x00\x00,\x00\x00\x00\x05\x00\x00\x00<\x00\x00\x00\x06\x00\x00\x00P\x00\x00\x00\x07\x00\x00\x00W\x00\x00\x00\x03\x00\x00\x00_\x00\x00\x00\x06\x00\x00\x00c\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00simple\x00unicode\x00Een\x00\xe2\x80\xa0wee\x00"


class TestMOUnit(test_base.TestTranslationUnit):
    UnitClass = mo.mounit

    def test_context(self):
        unit = self.UnitClass("Message")
        unit.setcontext("context")
        assert unit.getcontext() == "context"


posources = [
    r"""
msgid ""
msgstr ""
"PO-Revision-Date: 2006-02-09 23:33+0200\n"
"MIME-Version: 1.0\n"
"Content-Type: text/plain; charset=UTF-8\n"
"Content-Transfer-Encoding: 8-bit\n"
""",
    r"""
msgid ""
msgstr ""
"PO-Revision-Date: 2006-02-09 23:33+0200\n"
"MIME-Version: 1.0\n"
"Content-Type: text/plain; charset=UTF-8\n"
"Content-Transfer-Encoding: 8-bit\n"

msgid "plant"
msgstr ""
""",
    # The following test is commented out, because the hash-size is different
    # compared to gettext, since we're not counting untranslated units.
    # r'''
    # msgid ""
    # msgstr ""
    # "PO-Revision-Date: 2006-02-09 23:33+0200\n"
    # "MIME-Version: 1.0\n"
    # "Content-Type: text/plain; charset=UTF-8\n"
    # "Content-Transfer-Encoding: 8-bit\n"
    #
    # msgid "plant"
    # msgstr ""
    #
    # msgid ""
    # "_: Noun\n"
    # "convert"
    # msgstr "bekeerling"
    #''',
    r"""
msgid ""
msgstr ""
"PO-Revision-Date: 2006-02-09 23:33+0200\n"
"MIME-Version: 1.0\n"
"Content-Type: text/plain; charset=UTF-8\n"
"Content-Transfer-Encoding: 8-bit\n"

msgid "plant"
msgstr ""

msgid ""
"_: Noun\n"
"convert"
msgstr "bekeerling"

msgctxt "verb"
msgid ""
"convert"
msgstr "omskakel"
""",
    r"""
msgid ""
msgstr ""
"PO-Revision-Date: 2006-02-09 23:33+0200\n"
"MIME-Version: 1.0\n"
"Content-Type: text/plain; charset=UTF-8\n"
"Content-Transfer-Encoding: 8-bit\n"

msgid "plant"
msgstr ""

msgid ""
"_: Noun\n"
"convert"
msgstr "bekeerling"

msgctxt "verb"
msgid ""
"convert"
msgstr "omskakel"

msgid "tree"
msgid_plural "trees"
msgstr[0] ""
""",
    r"""
msgid ""
msgstr ""
"PO-Revision-Date: 2020-12-23 22:11+0000\n"
"MIME-Version: 1.0\n"
"Content-Type: text/plain; charset=UTF-8\n"
"Content-Transfer-Encoding: 8bit\n"

msgid "Contains %d project"
msgstr "Obsahuje %d projekt"
""",
    r"""
msgid ""
msgstr ""
"POT-Creation-Date: 2020-12-25 09:38+0000\n"
"PO-Revision-Date: 2020-12-23 22:11+0000\n"
"MIME-Version: 1.0\n"
"Content-Type: text/plain; charset=UTF-8\n"
"Content-Transfer-Encoding: 8bit\n"

#: weblate/accounts/avatar.py:103
msgctxt "No known user"
msgid "None"
msgstr "Žádný"
""",
    r"""
msgid ""
msgstr ""
"PO-Revision-Date: 2020-12-23 22:11+0000\n"
"MIME-Version: 1.0\n"
"Content-Type: text/plain; charset=UTF-8\n"
"Content-Transfer-Encoding: 8bit\n"
"Plural-Forms: nplurals=3; plural=(n==1) ? 0 : (n>=2 && n<=4) ? 1 : 2;\n"

#, python-format
msgid "%(count)s screenshot"
msgid_plural "%(count)s screenshots"
msgstr[0] "%(count)s obrázek"
msgstr[1] "%(count)s obrázky"
msgstr[2] "%(count)s obrázků"
""",
    r"""
msgid ""
msgstr ""
"MIME-Version: 1.0\n"
"Content-Type: text/plain; charset=UTF-8\n"
"Content-Transfer-Encoding: 8bit\n"

msgid "0"
msgstr "0"

msgid "1"
msgstr "1"

msgid "2"
msgstr "2"

msgid "3"
msgstr "3"

msgid "4"
msgstr "4"

msgid "5"
msgstr "5"

msgid "6"
msgstr "6"

msgid "7"
msgstr "7"

msgid "8"
msgstr "8"

msgid "9"
msgstr "9"

msgid "10"
msgstr "10"

msgid "11"
msgstr "11"

msgid "12"
msgstr "12"

msgid "13"
msgstr "13"

msgid "14"
msgstr "14"

msgid "15"
msgstr "15"

msgid "16"
msgstr "16"

msgid "17"
msgstr "17"

msgid "18"
msgstr "18"

msgid "19"
msgstr "19"

msgid "20"
msgstr "20"

msgid "21"
msgstr "21"

msgid "22"
msgstr "22"

msgid "23"
msgstr "23"

msgid "24"
msgstr "24"

msgid "25"
msgstr "25"

msgid "26"
msgstr "26"

msgid "27"
msgstr "27"

msgid "28"
msgstr "28"

msgid "29"
msgstr "29"

msgid "30"
msgstr "30"

msgid "31"
msgstr "31"

msgid "32"
msgstr "32"

msgid "33"
msgstr "33"

msgid "34"
msgstr "34"

msgid "35"
msgstr "35"

msgid "36"
msgstr "36"

msgid "37"
msgstr "37"

msgid "38"
msgstr "38"

msgid "39"
msgstr "39"

msgid "40"
msgstr "40"

msgid "41"
msgstr "41"

msgid "42"
msgstr "42"

msgid "43"
msgstr "43"

msgid "44"
msgstr "44"

msgid "45"
msgstr "45"

msgid "46"
msgstr "46"

msgid "47"
msgstr "47"

msgid "48"
msgstr "48"

msgid "49"
msgstr "49"

msgid "50"
msgstr "50"

msgid "51"
msgstr "51"

msgid "52"
msgstr "52"

msgid "53"
msgstr "53"

msgid "54"
msgstr "54"

msgid "55"
msgstr "55"

msgid "56"
msgstr "56"

msgid "57"
msgstr "57"

msgid "58"
msgstr "58"

msgid "59"
msgstr "59"

msgid "60"
msgstr "60"

msgid "61"
msgstr "61"

msgid "62"
msgstr "62"

msgid "63"
msgstr "63"

msgid "64"
msgstr "64"

msgid "65"
msgstr "65"

msgid "66"
msgstr "66"

msgid "67"
msgstr "67"

msgid "68"
msgstr "68"

msgid "69"
msgstr "69"

msgid "70"
msgstr "70"

msgid "71"
msgstr "71"

msgid "72"
msgstr "72"

msgid "73"
msgstr "73"

msgid "74"
msgstr "74"

msgid "75"
msgstr "75"

msgid "76"
msgstr "76"

msgid "77"
msgstr "77"

msgid "78"
msgstr "78"

msgid "79"
msgstr "79"

msgid "80"
msgstr "80"

msgid "81"
msgstr "81"

msgid "82"
msgstr "82"

msgid "83"
msgstr "83"

msgid "84"
msgstr "84"

msgid "85"
msgstr "85"

msgid "86"
msgstr "86"

msgid "87"
msgstr "87"

msgid "88"
msgstr "88"

msgid "89"
msgstr "89"

msgid "90"
msgstr "90"

msgid "91"
msgstr "91"

msgid "92"
msgstr "92"

msgid "93"
msgstr "93"

msgid "94"
msgstr "94"

msgid "95"
msgstr "95"

msgid "96"
msgstr "96"

msgid "97"
msgstr "97"

msgid "98"
msgstr "98"

msgid "99"
msgstr "99"
""",
]

mosources = [
    b"\xde\x12\x04\x95\x00\x00\x00\x00\x01\x00\x00\x00\x1c\x00\x00\x00$\x00\x00\x00\x03\x00\x00\x00,\x00\x00\x00\x00\x00\x00\x008\x00\x00\x00\x83\x00\x00\x009\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00PO-Revision-Date: 2006-02-09 23:33+0200\nMIME-Version: 1.0\nContent-Type: text/plain; charset=UTF-8\nContent-Transfer-Encoding: 8-bit\n\x00",
    b"\xde\x12\x04\x95\x00\x00\x00\x00\x01\x00\x00\x00\x1c\x00\x00\x00$\x00\x00\x00\x03\x00\x00\x00,\x00\x00\x00\x00\x00\x00\x008\x00\x00\x00\x83\x00\x00\x009\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00PO-Revision-Date: 2006-02-09 23:33+0200\nMIME-Version: 1.0\nContent-Type: text/plain; charset=UTF-8\nContent-Transfer-Encoding: 8-bit\n\x00",
    b"\xde\x12\x04\x95\x00\x00\x00\x00\x03\x00\x00\x00\x1c\x00\x00\x004\x00\x00\x00\x05\x00\x00\x00L\x00\x00\x00\x00\x00\x00\x00`\x00\x00\x00\x0f\x00\x00\x00a\x00\x00\x00\x0c\x00\x00\x00q\x00\x00\x00\x83\x00\x00\x00~\x00\x00\x00\n\x00\x00\x00\x02\x01\x00\x00\x08\x00\x00\x00\r\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x03\x00\x00\x00\x00_: Noun\nconvert\x00verb\x04convert\x00PO-Revision-Date: 2006-02-09 23:33+0200\nMIME-Version: 1.0\nContent-Type: text/plain; charset=UTF-8\nContent-Transfer-Encoding: 8-bit\n\x00bekeerling\x00omskakel\x00",
    b"\xde\x12\x04\x95\x00\x00\x00\x00\x03\x00\x00\x00\x1c\x00\x00\x004\x00\x00\x00\x05\x00\x00\x00L\x00\x00\x00\x00\x00\x00\x00`\x00\x00\x00\x0f\x00\x00\x00a\x00\x00\x00\x0c\x00\x00\x00q\x00\x00\x00\x83\x00\x00\x00~\x00\x00\x00\n\x00\x00\x00\x02\x01\x00\x00\x08\x00\x00\x00\r\x01\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x03\x00\x00\x00\x00_: Noun\nconvert\x00verb\x04convert\x00PO-Revision-Date: 2006-02-09 23:33+0200\nMIME-Version: 1.0\nContent-Type: text/plain; charset=UTF-8\nContent-Transfer-Encoding: 8-bit\n\x00bekeerling\x00omskakel\x00",
    b"\xde\x12\x04\x95\x00\x00\x00\x00\x02\x00\x00\x00\x1c\x00\x00\x00,\x00\x00\x00\x05\x00\x00\x00<\x00\x00\x00\x00\x00\x00\x00P\x00\x00\x00\x13\x00\x00\x00Q\x00\x00\x00\x82\x00\x00\x00e\x00\x00\x00\x13\x00\x00\x00\xe8\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00Contains %d project\x00PO-Revision-Date: 2020-12-23 22:11+0000\nMIME-Version: 1.0\nContent-Type: text/plain; charset=UTF-8\nContent-Transfer-Encoding: 8bit\n\x00Obsahuje %d projekt\x00",
    b"\xde\x12\x04\x95\x00\x00\x00\x00\x02\x00\x00\x00\x1c\x00\x00\x00,\x00\x00\x00\x05\x00\x00\x00<\x00\x00\x00\x00\x00\x00\x00P\x00\x00\x00\x12\x00\x00\x00Q\x00\x00\x00\x82\x00\x00\x00d\x00\x00\x00\x08\x00\x00\x00\xe7\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00No known user\x04None\x00PO-Revision-Date: 2020-12-23 22:11+0000\nMIME-Version: 1.0\nContent-Type: text/plain; charset=UTF-8\nContent-Transfer-Encoding: 8bit\n\x00\xc5\xbd\xc3\xa1dn\xc3\xbd\x00",
    b"\xde\x12\x04\x95\x00\x00\x00\x00\x02\x00\x00\x00\x1c\x00\x00\x00,\x00\x00\x00\x05\x00\x00\x00<\x00\x00\x00\x00\x00\x00\x00P\x00\x00\x00*\x00\x00\x00Q\x00\x00\x00\xc8\x00\x00\x00|\x00\x00\x009\x00\x00\x00E\x01\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00%(count)s screenshot\x00%(count)s screenshots\x00PO-Revision-Date: 2020-12-23 22:11+0000\nMIME-Version: 1.0\nContent-Type: text/plain; charset=UTF-8\nContent-Transfer-Encoding: 8bit\nPlural-Forms: nplurals=3; plural=(n==1) ? 0 : (n>=2 && n<=4) ? 1 : 2;\n\x00%(count)s obr\xc3\xa1zek\x00%(count)s obr\xc3\xa1zky\x00%(count)s obr\xc3\xa1zk\xc5\xaf\x00",
    b"\xde\x12\x04\x95\x00\x00\x00\x00e\x00\x00\x00\x1c\x00\x00\x00D\x03\x00\x00\x89\x00\x00\x00l\x06\x00\x00\x00\x00\x00\x00\x90\x08\x00\x00\x01\x00\x00\x00\x91\x08\x00\x00\x01\x00\x00\x00\x93\x08\x00\x00\x02\x00\x00\x00\x95\x08\x00\x00\x02\x00\x00\x00\x98\x08\x00\x00\x02\x00\x00\x00\x9b\x08\x00\x00\x02\x00\x00\x00\x9e\x08\x00\x00\x02\x00\x00\x00\xa1\x08\x00\x00\x02\x00\x00\x00\xa4\x08\x00\x00\x02\x00\x00\x00\xa7\x08\x00\x00\x02\x00\x00\x00\xaa\x08\x00\x00\x02\x00\x00\x00\xad\x08\x00\x00\x02\x00\x00\x00\xb0\x08\x00\x00\x01\x00\x00\x00\xb3\x08\x00\x00\x02\x00\x00\x00\xb5\x08\x00\x00\x02\x00\x00\x00\xb8\x08\x00\x00\x02\x00\x00\x00\xbb\x08\x00\x00\x02\x00\x00\x00\xbe\x08\x00\x00\x02\x00\x00\x00\xc1\x08\x00\x00\x02\x00\x00\x00\xc4\x08\x00\x00\x02\x00\x00\x00\xc7\x08\x00\x00\x02\x00\x00\x00\xca\x08\x00\x00\x02\x00\x00\x00\xcd\x08\x00\x00\x02\x00\x00\x00\xd0\x08\x00\x00\x01\x00\x00\x00\xd3\x08\x00\x00\x02\x00\x00\x00\xd5\x08\x00\x00\x02\x00\x00\x00\xd8\x08\x00\x00\x02\x00\x00\x00\xdb\x08\x00\x00\x02\x00\x00\x00\xde\x08\x00\x00\x02\x00\x00\x00\xe1\x08\x00\x00\x02\x00\x00\x00\xe4\x08\x00\x00\x02\x00\x00\x00\xe7\x08\x00\x00\x02\x00\x00\x00\xea\x08\x00\x00\x02\x00\x00\x00\xed\x08\x00\x00\x02\x00\x00\x00\xf0\x08\x00\x00\x01\x00\x00\x00\xf3\x08\x00\x00\x02\x00\x00\x00\xf5\x08\x00\x00\x02\x00\x00\x00\xf8\x08\x00\x00\x02\x00\x00\x00\xfb\x08\x00\x00\x02\x00\x00\x00\xfe\x08\x00\x00\x02\x00\x00\x00\x01\t\x00\x00\x02\x00\x00\x00\x04\t\x00\x00\x02\x00\x00\x00\x07\t\x00\x00\x02\x00\x00\x00\n\t\x00\x00\x02\x00\x00\x00\r\t\x00\x00\x02\x00\x00\x00\x10\t\x00\x00\x01\x00\x00\x00\x13\t\x00\x00\x02\x00\x00\x00\x15\t\x00\x00\x02\x00\x00\x00\x18\t\x00\x00\x02\x00\x00\x00\x1b\t\x00\x00\x02\x00\x00\x00\x1e\t\x00\x00\x02\x00\x00\x00!\t\x00\x00\x02\x00\x00\x00$\t\x00\x00\x02\x00\x00\x00'\t\x00\x00\x02\x00\x00\x00*\t\x00\x00\x02\x00\x00\x00-\t\x00\x00\x02\x00\x00\x000\t\x00\x00\x01\x00\x00\x003\t\x00\x00\x02\x00\x00\x005\t\x00\x00\x02\x00\x00\x008\t\x00\x00\x02\x00\x00\x00;\t\x00\x00\x02\x00\x00\x00>\t\x00\x00\x02\x00\x00\x00A\t\x00\x00\x02\x00\x00\x00D\t\x00\x00\x02\x00\x00\x00G\t\x00\x00\x02\x00\x00\x00J\t\x00\x00\x02\x00\x00\x00M\t\x00\x00\x02\x00\x00\x00P\t\x00\x00\x01\x00\x00\x00S\t\x00\x00\x02\x00\x00\x00U\t\x00\x00\x02\x00\x00\x00X\t\x00\x00\x02\x00\x00\x00[\t\x00\x00\x02\x00\x00\x00^\t\x00\x00\x02\x00\x00\x00a\t\x00\x00\x02\x00\x00\x00d\t\x00\x00\x02\x00\x00\x00g\t\x00\x00\x02\x00\x00\x00j\t\x00\x00\x02\x00\x00\x00m\t\x00\x00\x02\x00\x00\x00p\t\x00\x00\x01\x00\x00\x00s\t\x00\x00\x02\x00\x00\x00u\t\x00\x00\x02\x00\x00\x00x\t\x00\x00\x02\x00\x00\x00{\t\x00\x00\x02\x00\x00\x00~\t\x00\x00\x02\x00\x00\x00\x81\t\x00\x00\x02\x00\x00\x00\x84\t\x00\x00\x02\x00\x00\x00\x87\t\x00\x00\x02\x00\x00\x00\x8a\t\x00\x00\x02\x00\x00\x00\x8d\t\x00\x00\x02\x00\x00\x00\x90\t\x00\x00\x01\x00\x00\x00\x93\t\x00\x00\x02\x00\x00\x00\x95\t\x00\x00\x02\x00\x00\x00\x98\t\x00\x00\x02\x00\x00\x00\x9b\t\x00\x00\x02\x00\x00\x00\x9e\t\x00\x00\x02\x00\x00\x00\xa1\t\x00\x00\x02\x00\x00\x00\xa4\t\x00\x00\x02\x00\x00\x00\xa7\t\x00\x00\x02\x00\x00\x00\xaa\t\x00\x00\x02\x00\x00\x00\xad\t\x00\x00\x02\x00\x00\x00\xb0\t\x00\x00Z\x00\x00\x00\xb3\t\x00\x00\x01\x00\x00\x00\x0e\n\x00\x00\x01\x00\x00\x00\x10\n\x00\x00\x02\x00\x00\x00\x12\n\x00\x00\x02\x00\x00\x00\x15\n\x00\x00\x02\x00\x00\x00\x18\n\x00\x00\x02\x00\x00\x00\x1b\n\x00\x00\x02\x00\x00\x00\x1e\n\x00\x00\x02\x00\x00\x00!\n\x00\x00\x02\x00\x00\x00$\n\x00\x00\x02\x00\x00\x00'\n\x00\x00\x02\x00\x00\x00*\n\x00\x00\x02\x00\x00\x00-\n\x00\x00\x01\x00\x00\x000\n\x00\x00\x02\x00\x00\x002\n\x00\x00\x02\x00\x00\x005\n\x00\x00\x02\x00\x00\x008\n\x00\x00\x02\x00\x00\x00;\n\x00\x00\x02\x00\x00\x00>\n\x00\x00\x02\x00\x00\x00A\n\x00\x00\x02\x00\x00\x00D\n\x00\x00\x02\x00\x00\x00G\n\x00\x00\x02\x00\x00\x00J\n\x00\x00\x02\x00\x00\x00M\n\x00\x00\x01\x00\x00\x00P\n\x00\x00\x02\x00\x00\x00R\n\x00\x00\x02\x00\x00\x00U\n\x00\x00\x02\x00\x00\x00X\n\x00\x00\x02\x00\x00\x00[\n\x00\x00\x02\x00\x00\x00^\n\x00\x00\x02\x00\x00\x00a\n\x00\x00\x02\x00\x00\x00d\n\x00\x00\x02\x00\x00\x00g\n\x00\x00\x02\x00\x00\x00j\n\x00\x00\x02\x00\x00\x00m\n\x00\x00\x01\x00\x00\x00p\n\x00\x00\x02\x00\x00\x00r\n\x00\x00\x02\x00\x00\x00u\n\x00\x00\x02\x00\x00\x00x\n\x00\x00\x02\x00\x00\x00{\n\x00\x00\x02\x00\x00\x00~\n\x00\x00\x02\x00\x00\x00\x81\n\x00\x00\x02\x00\x00\x00\x84\n\x00\x00\x02\x00\x00\x00\x87\n\x00\x00\x02\x00\x00\x00\x8a\n\x00\x00\x02\x00\x00\x00\x8d\n\x00\x00\x01\x00\x00\x00\x90\n\x00\x00\x02\x00\x00\x00\x92\n\x00\x00\x02\x00\x00\x00\x95\n\x00\x00\x02\x00\x00\x00\x98\n\x00\x00\x02\x00\x00\x00\x9b\n\x00\x00\x02\x00\x00\x00\x9e\n\x00\x00\x02\x00\x00\x00\xa1\n\x00\x00\x02\x00\x00\x00\xa4\n\x00\x00\x02\x00\x00\x00\xa7\n\x00\x00\x02\x00\x00\x00\xaa\n\x00\x00\x02\x00\x00\x00\xad\n\x00\x00\x01\x00\x00\x00\xb0\n\x00\x00\x02\x00\x00\x00\xb2\n\x00\x00\x02\x00\x00\x00\xb5\n\x00\x00\x02\x00\x00\x00\xb8\n\x00\x00\x02\x00\x00\x00\xbb\n\x00\x00\x02\x00\x00\x00\xbe\n\x00\x00\x02\x00\x00\x00\xc1\n\x00\x00\x02\x00\x00\x00\xc4\n\x00\x00\x02\x00\x00\x00\xc7\n\x00\x00\x02\x00\x00\x00\xca\n\x00\x00\x02\x00\x00\x00\xcd\n\x00\x00\x01\x00\x00\x00\xd0\n\x00\x00\x02\x00\x00\x00\xd2\n\x00\x00\x02\x00\x00\x00\xd5\n\x00\x00\x02\x00\x00\x00\xd8\n\x00\x00\x02\x00\x00\x00\xdb\n\x00\x00\x02\x00\x00\x00\xde\n\x00\x00\x02\x00\x00\x00\xe1\n\x00\x00\x02\x00\x00\x00\xe4\n\x00\x00\x02\x00\x00\x00\xe7\n\x00\x00\x02\x00\x00\x00\xea\n\x00\x00\x02\x00\x00\x00\xed\n\x00\x00\x01\x00\x00\x00\xf0\n\x00\x00\x02\x00\x00\x00\xf2\n\x00\x00\x02\x00\x00\x00\xf5\n\x00\x00\x02\x00\x00\x00\xf8\n\x00\x00\x02\x00\x00\x00\xfb\n\x00\x00\x02\x00\x00\x00\xfe\n\x00\x00\x02\x00\x00\x00\x01\x0b\x00\x00\x02\x00\x00\x00\x04\x0b\x00\x00\x02\x00\x00\x00\x07\x0b\x00\x00\x02\x00\x00\x00\n\x0b\x00\x00\x02\x00\x00\x00\r\x0b\x00\x00\x01\x00\x00\x00\x10\x0b\x00\x00\x02\x00\x00\x00\x12\x0b\x00\x00\x02\x00\x00\x00\x15\x0b\x00\x00\x02\x00\x00\x00\x18\x0b\x00\x00\x02\x00\x00\x00\x1b\x0b\x00\x00\x02\x00\x00\x00\x1e\x0b\x00\x00\x02\x00\x00\x00!\x0b\x00\x00\x02\x00\x00\x00$\x0b\x00\x00\x02\x00\x00\x00'\x0b\x00\x00\x02\x00\x00\x00*\x0b\x00\x00\x02\x00\x00\x00-\x0b\x00\x00\x01\x00\x00\x00\\\x00\x00\x00]\x00\x00\x00^\x00\x00\x00I\x00\x00\x00`\x00\x00\x00a\x00\x00\x00b\x00\x00\x00c\x00\x00\x00d\x00\x00\x00\x04\x00\x00\x00\x05\x00\x00\x00\x06\x00\x00\x00\x07\x00\x00\x00\x08\x00\x00\x00\t\x00\x00\x00\n\x00\x00\x00\x0b\x00\x00\x00\x0c\x00\x00\x00\r\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00_\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0f\x00\x00\x00\x10\x00\x00\x00\x11\x00\x00\x00\x12\x00\x00\x00\x13\x00\x00\x00\x14\x00\x00\x00\x15\x00\x00\x00\x16\x00\x00\x00\x17\x00\x00\x00\x18\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x1a\x00\x00\x00\x1b\x00\x00\x00\x1c\x00\x00\x00\x1d\x00\x00\x00\x1e\x00\x00\x00\x1f\x00\x00\x00\x02\x00\x00\x00\x03\x00\x00\x00\x0e\x00\x00\x00\x19\x00\x00\x00$\x00\x00\x00/\x00\x00\x00:\x00\x00\x00E\x00\x00\x00P\x00\x00\x00[\x00\x00\x00%\x00\x00\x00&\x00\x00\x00'\x00\x00\x00(\x00\x00\x00)\x00\x00\x00*\x00\x00\x00+\x00\x00\x00,\x00\x00\x00-\x00\x00\x00.\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x000\x00\x00\x001\x00\x00\x002\x00\x00\x003\x00\x00\x004\x00\x00\x005\x00\x00\x006\x00\x00\x007\x00\x00\x008\x00\x00\x009\x00\x00\x00\x00\x00\x00\x00K\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00O\x00\x00\x00\x00\x00\x00\x00;\x00\x00\x00<\x00\x00\x00=\x00\x00\x00>\x00\x00\x00?\x00\x00\x00@\x00\x00\x00A\x00\x00\x00B\x00\x00\x00C\x00\x00\x00D\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00M\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00F\x00\x00\x00G\x00\x00\x00H\x00\x00\x00 \x00\x00\x00J\x00\x00\x00!\x00\x00\x00L\x00\x00\x00\"\x00\x00\x00N\x00\x00\x00#\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00Q\x00\x00\x00R\x00\x00\x00S\x00\x00\x00T\x00\x00\x00U\x00\x00\x00V\x00\x00\x00W\x00\x00\x00X\x00\x00\x00Y\x00\x00\x00Z\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00e\x00\x00\x00\x00\x00\x00\x00\x000\x001\x0010\x0011\x0012\x0013\x0014\x0015\x0016\x0017\x0018\x0019\x002\x0020\x0021\x0022\x0023\x0024\x0025\x0026\x0027\x0028\x0029\x003\x0030\x0031\x0032\x0033\x0034\x0035\x0036\x0037\x0038\x0039\x004\x0040\x0041\x0042\x0043\x0044\x0045\x0046\x0047\x0048\x0049\x005\x0050\x0051\x0052\x0053\x0054\x0055\x0056\x0057\x0058\x0059\x006\x0060\x0061\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069\x007\x0070\x0071\x0072\x0073\x0074\x0075\x0076\x0077\x0078\x0079\x008\x0080\x0081\x0082\x0083\x0084\x0085\x0086\x0087\x0088\x0089\x009\x0090\x0091\x0092\x0093\x0094\x0095\x0096\x0097\x0098\x0099\x00MIME-Version: 1.0\nContent-Type: text/plain; charset=UTF-8\nContent-Transfer-Encoding: 8bit\n\x000\x001\x0010\x0011\x0012\x0013\x0014\x0015\x0016\x0017\x0018\x0019\x002\x0020\x0021\x0022\x0023\x0024\x0025\x0026\x0027\x0028\x0029\x003\x0030\x0031\x0032\x0033\x0034\x0035\x0036\x0037\x0038\x0039\x004\x0040\x0041\x0042\x0043\x0044\x0045\x0046\x0047\x0048\x0049\x005\x0050\x0051\x0052\x0053\x0054\x0055\x0056\x0057\x0058\x0059\x006\x0060\x0061\x0062\x0063\x0064\x0065\x0066\x0067\x0068\x0069\x007\x0070\x0071\x0072\x0073\x0074\x0075\x0076\x0077\x0078\x0079\x008\x0080\x0081\x0082\x0083\x0084\x0085\x0086\x0087\x0088\x0089\x009\x0090\x0091\x0092\x0093\x0094\x0095\x0096\x0097\x0098\x0099\x00",
]


class TestMOFile(test_base.TestTranslationStore):
    StoreClass = mo.mofile

    def get_mo_and_po(self):
        return (
            os.path.abspath(self.filename + ".po"),
            os.path.abspath(self.filename + ".msgfmt.mo"),
            os.path.abspath(self.filename + ".pocompile.mo"),
        )

    def remove_po_and_mo(self):
        for file in self.get_mo_and_po():
            if os.path.exists(file):
                os.remove(file)

    def setup_method(self, method):
        super().setup_method(method)
        self.remove_po_and_mo()

    def teardown_method(self, method):
        super().teardown_method(method)
        self.remove_po_and_mo()

    def test_language(self):
        """Test that we can return the target language correctly."""
        store = self.StoreClass()
        store.updateheader(add=True, Language="zu")
        assert store.gettargetlanguage() == "zu"

    def test_context(self):
        store = self.StoreClass()
        unit = self.StoreClass.UnitClass("source")
        unit.target = "target"
        unit.setcontext("context")
        store.addunit(unit)
        assert b"context" in (bytes(store))

        newstore = self.StoreClass.parsestring(bytes(store))
        assert len(newstore.units) == 1
        assert newstore.units[0].getcontext(), "context"

    def test_output(self):
        assert len(posources) == len(mosources), "Missing MO rendering for PO sources"
        for i, posource in enumerate(posources):
            print("PO source file")
            print(posource)
            if "POT-Creation-Date" in posource and os.name == "nt":
                # There is no Gettext 0.20 on Windows, so the output is different
                continue
            PO_FILE, MO_MSGFMT, MO_POCOMPILE = self.get_mo_and_po()
            posource = posource.encode("utf-8")

            with open(PO_FILE, "wb") as out_file:
                out_file.write(posource)

            subprocess.check_call(
                ["msgfmt", PO_FILE, "-o", MO_MSGFMT, "--endianness", sys.byteorder]
            )
            argv = sys.argv
            try:
                sys.argv = [
                    "pocompile",
                    "--errorlevel=traceback",
                    PO_FILE,
                    MO_POCOMPILE,
                ]
                pocompile.main()
            finally:
                sys.argv = argv

            store = factory.getobject(BytesIO(posource))
            if store.isempty() and not os.path.exists(MO_POCOMPILE):
                # pocompile doesn't create MO files for empty PO files, so we
                # can skip the checks here.
                continue

            with open(MO_MSGFMT, "rb") as mo_msgfmt_f:
                mo_msgfmt = mo_msgfmt_f.read()
            print("msgfmt output:")
            print(repr(mo_msgfmt))

            with open(MO_POCOMPILE, "rb") as mo_pocompile_f:
                mo_pocompile = mo_pocompile_f.read()
            print("pocompile output:")
            print(repr(mo_pocompile))

            # Compare to known values on little endian system as that is the
            # endianness used in the test data.
            if sys.byteorder == "little":
                assert mo_pocompile == mosources[i]

            # Do not compare whole content because Gettext has changed the
            # hashing table content in 0.25

            # Extract header
            msgfmt_header = self.StoreClass.parse_header(mo_msgfmt)
            pocompile_header = self.StoreClass.parse_header(mo_pocompile)
            # Compare header without hash table sizes
            assert msgfmt_header[:-2] == pocompile_header[:-2], "MO header mismatch"

            # The offsets and hash table will differ

            # Compare string data (after hash table)
            assert (
                mo_msgfmt[msgfmt_header[7] + msgfmt_header[6] * 4 :]
                == mo_pocompile[pocompile_header[7] + pocompile_header[6] * 4 :]
            )

            # Verify parsing of generated files
            self.StoreClass.parsefile(MO_POCOMPILE)
            self.StoreClass.parsefile(MO_MSGFMT)

    def test_endian_version_parsing(self) -> None:
        store_big = self.StoreClass(MO_BIG_ENDIAN)
        store_little = self.StoreClass(MO_LITTLE_ENDIAN)
        assert store_big.units == store_little.units
